﻿const pewModule = require('./pew_module');
const pew = new pewModule.pewModule();
const Mew = require('./mewtocol');
const Mew7 = require('./mewtocol7');
const Modbus = require('./modbus');
const fs = require("fs");
const path = require('path');
var db, client;
var sqlClientRunning = false;

const moduleName = "SQL Client";

//Catch exceptions
process.on('uncaughtException', function (err) {
    pew.sysLogMessage(moduleName, "Uncaught exception: " + err.stack);
    if (sqlClientRunning) {
        done();
    }
})

//Read Configurations
var interfaceConfig = pew.getConfFileSync(pew.Constants.requests.READ_INTERFACE);
var portConfig = pew.getConfFileSync(pew.Constants.requests.READ_PORTS);
var sqlConfig = pew.getConfFileSync(pew.Constants.requests.READ_SQL);

//If any configuration could not be loaded -> log the message and stop execution
if (interfaceConfig.err) {
    pew.sysLogMessage(moduleName, interfaceConfig.err_msg);
    return;
}

if (portConfig.err) {
    pew.sysLogMessage(moduleName, portConfig.err_msg);
    return;
}

if (sqlConfig.err) {
    pew.sysLogMessage(moduleName, sqlConfig.err_msg);
    return;
}

//Check configurations
var usePort, useProtocol, serviceID, protocol;
//Check wheather to use with or without PLC
serviceID = sqlConfig.data.insertWithoutPLC ? sqlConfig.data.interface : interfaceConfig.data.interface_plc;

let iface = interfaceConfig.data.interface.filter(val => {
    return val.service == serviceID;
})

if (iface.length <= 0 || !iface[0].enabled) {
    return pew.sysLogMessage(moduleName, pew.Constants.COM_NOT_ENABLED);
}

useProtocol = iface[0].protocol;

//Set the protocol module
switch (useProtocol) {
    case pew.Constants.PROTOCOLS.mewtocol:
        protocol = new Mew.Mewtocol();
        break;

    case pew.Constants.PROTOCOLS.mewtocol7:
        protocol = new Mew7.Mewtocol7();
        break;

    case pew.Constants.PROTOCOLS.modbus:
        protocol = new Modbus.Modbus();
        break;

    default:
        protocol = mewtocol;
}

var thread = portConfig.data.thread.filter(val => {
    return val.service === serviceID;
});

if (thread.length <= 0) {
    return pew.sysLogMessage(moduleName, pew.Constants.TCP_PORT_DISABLED);
}
usePort = thread[0].port;

//Configuration all OK -> continue
let STATUS = {
    OK: 0,
    generalErr: -1,
    connectTimeout: -2
}

let dbTypes = {
    mssql: 0,
    mysql: 1,
    postgres: 2,
    mariadb: 3,
    azure: 4
}

setupSQLConnection();

function setupSQLConnection() {
    pew.getKeyContent().then(key => {
        pew.Constants.pwkey = key;
        switch (sqlConfig.data.db_type) {
            case dbTypes.mssql:
            case dbTypes.azure:
                db = require("mssql");
                client = {
                    user: sqlConfig.data.user,
                    password: sqlConfig.data.pass === "" ? sqlConfig.data.pass : pew.decryptPW(sqlConfig.data.pass),
                    database: sqlConfig.data.instance_name,
                    server: sqlConfig.data.server,
                    port: sqlConfig.data.port,
                    connectionTimeout: sqlConfig.data.timeout,
                    options: {
                        encrypt: sqlConfig.data.db_type === dbTypes.azure ? true : !!sqlConfig.data.use_encryption, // for azure
                        trustServerCertificate: sqlConfig.data.db_type === dbTypes.azure ? false : !sqlConfig.data.ca_verification, // change to true for local dev / self-signed certs
                    },
                }
                break;

            case dbTypes.mysql:
            case dbTypes.mariadb:
                db = require("mysql");
                client = db.createPool({
                    host: sqlConfig.data.server,
                    user: sqlConfig.data.user,
                    password: sqlConfig.data.pass === "" ? sqlConfig.data.pass : pew.decryptPW(sqlConfig.data.pass),
                    database: sqlConfig.data.instance_name,
                    port: sqlConfig.data.port,
                    connectTimeout: sqlConfig.data.timeout,
                    ssl: sqlConfig.data.use_encryption ? {
                        rejectUnauthorized: sqlConfig.data.ca_verification,
                        ca: sqlConfig.data.ca !== pew.Constants.EMPTY_STRING ? fs.readFileSync(path.join(__dirname, pew.Constants.configSubfolders.sql, sqlConfig.data.ca)) : null,
                        cert: sqlConfig.data.client !== pew.Constants.EMPTY_STRING ? fs.readFileSync(path.join(__dirname, pew.Constants.configSubfolders.sql, sqlConfig.data.client)) : null,
                        key: sqlConfig.data.key !== pew.Constants.EMPTY_STRING ? fs.readFileSync(path.join(__dirname, pew.Constants.configSubfolders.sql, sqlConfig.data.key)) : null,
                    } : null
                })
                break;

            case dbTypes.postgres:
                db = require("pg");
                client = new db.Pool({
                    host: sqlConfig.data.server,
                    port: sqlConfig.data.port,
                    user: sqlConfig.data.user,
                    password: sqlConfig.data.pass === "" ? sqlConfig.data.pass : pew.decryptPW(sqlConfig.data.pass),
                    database: sqlConfig.data.instance_name,
                    connectionTimeoutMillis: sqlConfig.data.timeout,
                    ssl: sqlConfig.data.use_encryption ? {
                        rejectUnauthorized: sqlConfig.data.ca_verification,
                        ca: sqlConfig.data.ca !== pew.Constants.EMPTY_STRING ? fs.readFileSync(path.join(__dirname, pew.Constants.configSubfolders.sql, sqlConfig.data.ca)) : null,
                        cert: sqlConfig.data.client !== pew.Constants.EMPTY_STRING ? fs.readFileSync(path.join(__dirname, pew.Constants.configSubfolders.sql, sqlConfig.data.client)) : null,
                        key: sqlConfig.data.key !== pew.Constants.EMPTY_STRING ? fs.readFileSync(path.join(__dirname, pew.Constants.configSubfolders.sql, sqlConfig.data.key)) : null,
                    } : null
                })
                break;
        }

        plcPolling();
        sqlClientRunning = true;
    }).catch(ex => {
        console.log(`${moduleName}: Failed to read key for SQL Client: ${ex}\n`);
        setTimeout(setupSQLConnection, 2000);
    })
}

async function plcPolling() {
    let readItem = {
        area: "DT",
        type: "DINT",
        count: 3,
        start: sqlConfig.data.plc_control_dt
    }
    const modes = {
        generalMode: 1, //user defined command to send
        notUsed: 2,  //for compatibility the status 2 will not be used (used in plc program)
        insertData: 3,
        selectDataDINT: 4, //select data of type DINT
        selectDataUDINT: 5,
        selectDataREAL: 6
    }

    let stringLenMaxChars = 120;
    //Normal plc controlled commands
    if (!sqlConfig.data.insertWithoutPLC) {
        protocol.StartAddr = readItem.start;
        pew.readMultipleRegisters(protocol, usePort, readItem, true).then(data => {
            try {
                if (!data.err) {
                    let control = data.dint[0];
                    let item = {
                        area: "DT",
                        start: data.dint[1]
                    }
                    let writeStringAddr = data.dint[2];

                    switch (control) {
                        //General Mode (user sends his own individual sql command)
                        case modes.generalMode:
                        case modes.insertData:
                            //Read the max. writable string length of PLC
                            readItem.type = "INT";
                            readItem.count = 1;
                            readItem.start = data.dint[2];
                            protocol.StartAddr = readItem.start;
                            pew.readMultipleRegisters(protocol, usePort, readItem).then(maxStringLen => {
                                if (!maxStringLen.err) {
                                    stringLenMaxChars = maxStringLen.int[0];
                                    pew.readString(protocol, usePort, item).then(data => {
                                        if (!data.err) {
                                            if (data.string === pew.Constants.EMPTY_STRING) {
                                                data.string = data.string.replace(new RegExp('\"', "g"), '\'');;
                                                setPLCText(cutStringLen(pew.Constants.EMPTY_STRING_NOT_ALLOWED, stringLenMaxChars), writeStringAddr).catch(() => { }).finally(() => {
                                                    setPLCStatus(STATUS.generalErr).catch({}).finally(() => {
                                                        done();
                                                    });
                                                })
                                            }
                                            else {
                                                data.string = data.string.replace(new RegExp('\"', "g"), '\'');;
                                                switch (sqlConfig.data.db_type) {
                                                    case dbTypes.mssql:
                                                    case dbTypes.azure:
                                                        executeMSSQLCommand(data.string).then(result => {
                                                            //if the result.length is more than 0 it means it was a reading request
                                                            //otherwise it was either update or insert probably
                                                            let sendString = result.length !== typeof pew.Constants.UNDEFINED ? cutStringLen(JSON.stringify(result), stringLenMaxChars) : pew.Constants.OK;
                                                            setPLCText(sendString, writeStringAddr).catch(() => { }).finally(() => {
                                                                setPLCStatus(STATUS.OK).catch({}).finally(() => {
                                                                    done();
                                                                });
                                                            })
                                                        }).catch(err => {
                                                            setPLCText(cutStringLen(err, stringLenMaxChars), writeStringAddr).catch({}).finally(() => {
                                                                setPLCStatus(STATUS.generalErr).catch({}).finally(() => {
                                                                    done();
                                                                });
                                                            });
                                                        })
                                                        break;

                                                    case dbTypes.mysql:
                                                    case dbTypes.mariadb:
                                                        executeMySQLCommand(data.string).then(result => {
                                                            let sendString = result.length !== typeof pew.Constants.UNDEFINED ? cutStringLen(JSON.stringify(result), stringLenMaxChars) : pew.Constants.OK;
                                                            setPLCText(sendString, writeStringAddr).catch(() => { }).finally(() => {
                                                                setPLCStatus(STATUS.OK).catch({}).finally(() => {
                                                                    done();
                                                                });
                                                            })
                                                        }).catch(err => {
                                                            setPLCText(cutStringLen(err, stringLenMaxChars), writeStringAddr).catch({}).finally(() => {
                                                                setPLCStatus(STATUS.generalErr).catch({}).finally(() => {
                                                                    done();
                                                                });
                                                            });
                                                        })
                                                        break;

                                                    case dbTypes.postgres:
                                                        executePostgresCommand(data.string).then(result => {
                                                            let sendString = result.length !== typeof pew.Constants.UNDEFINED ? cutStringLen(JSON.stringify(result), stringLenMaxChars) : pew.Constants.OK;
                                                            setPLCText(sendString, writeStringAddr).catch(() => { }).finally(() => {
                                                                setPLCStatus(STATUS.OK).catch({}).finally(() => {
                                                                    done();
                                                                });
                                                            })
                                                        }).catch(err => {
                                                            setPLCText(cutStringLen(err, stringLenMaxChars), writeStringAddr).catch({}).finally(() => {
                                                                setPLCStatus(STATUS.generalErr).catch({}).finally(() => {
                                                                    done();
                                                                });
                                                            });
                                                        })
                                                        break;
                                                }
                                            }
                                        }
                                    }).catch(err => {
                                        pew.sysLogMessage(moduleName, err.err_msg);
                                        setPLCText(cutStringLen(err.err_msg, stringLenMaxChars)).catch({}).finally(() => {
                                            setPLCStatus(STATUS.generalErr).catch({}).finally(() => {
                                                done();
                                            });
                                        });
                                    })
                                }

                            }).catch(err => {
                                pew.sysLogMessage(moduleName, err.message);
                                setPLCText(cutStringLen(err.message, stringLenMaxChars)).catch({}).finally(() => {
                                    setPLCStatus(STATUS.generalErr).catch({}).finally(() => {
                                        done();
                                    });
                                });
                            })
                            break;

                        //Other modes....
                        case modes.selectDataDINT:
                        case modes.selectDataREAL:
                        case modes.selectDataUDINT:
                            pew.readString(protocol, usePort, item).then(data => {
                                if (!data.err) {
                                    let writeString;
                                    if (data.string === pew.Constants.EMPTY_STRING) {
                                        data.string = data.string.replace(new RegExp('\"', "g"), '\'');;
                                        writeString = cutStringLen(pew.Constants.EMPTY_STRING_NOT_ALLOWED, pew.Constants.MAX_SELECT_VALUES);
                                        setPLCText(writeString, writeStringAddr).catch(() => { }).finally(() => {
                                            setPLCStatus(STATUS.generalErr).catch({}).finally(() => {
                                                done();
                                            });
                                        })
                                    }
                                    else {
                                        data.string = data.string.replace(new RegExp('\"', "g"), '\'');;
                                        switch (sqlConfig.data.db_type) {
                                            case dbTypes.mssql:
                                            case dbTypes.azure:
                                                executeMSSQLCommand(data.string).then(result => {
                                                    //+2 data length, since we use the first DINT/UDINT/REAL as length
                                                    let dataLen = result.length * 2 + 2;
                                                    let dataWriteType;
                                                    let data = [];
                                                    let writeData = {
                                                        area: "DT",
                                                        count: dataLen,
                                                        start: writeStringAddr,
                                                    }

                                                    switch (control) {
                                                        case modes.selectDataDINT:
                                                            writeData.type = "DINT";
                                                            dataWriteType = pew.Constants.writeDataTypes._32bit;
                                                            break;

                                                        case modes.selectDataUDINT:
                                                            writeData.type = "UDINT";
                                                            dataWriteType = pew.Constants.writeDataTypes._32bit;
                                                            break;

                                                        case modes.selectDataREAL:
                                                            writeData.type = "REAL";
                                                            dataWriteType = pew.Constants.writeDataTypes._32bitReal;
                                                            break;
                                                    }

                                                    //Push the data length first to the first register
                                                    data.push(dataLen);

                                                    if (result.length && result.length > 0) {
                                                        for (let r of result) {
                                                            if (Object.values(r)[0] === null) {
                                                                dataLen -= 2;
                                                            }
                                                            else {
                                                                data.push(Object.values(r)[0]);
                                                            }
                                                        }
                                                    }
                                                    //Since the data length could have changed, update it
                                                    data[0] = dataLen;
                                                    writeData.count = dataLen / 2;
                                                    writeData.value = data;
                                                    pew.writeMultipleRegisters(protocol, usePort, writeData, dataWriteType)
                                                        .then(() => {
                                                            setPLCStatus(STATUS.OK).catch({}).finally(() => {
                                                                done();
                                                            });
                                                        })
                                                        .catch(err => {
                                                            let errorMessage;
                                                            errorMessage = err.err_msg ? err.err_msg : err;

                                                            pew.sysLogMessage(moduleName, errorMessage);
                                                            setPLCText(cutStringLen(errorMessage, pew.Constants.MAX_SELECT_VALUES), writeStringAddr).catch(() => { }).finally(() => {
                                                                setPLCStatus(STATUS.generalErr).catch({}).finally(() => {
                                                                    done();
                                                                });
                                                            })
                                                        });
                                                }).catch(err => {
                                                    setPLCText(cutStringLen(err, stringLenMaxChars), writeStringAddr).catch({}).finally(() => {
                                                        setPLCStatus(STATUS.generalErr).catch({}).finally(() => {
                                                            done();
                                                        });
                                                    });
                                                })
                                                break;

                                            case dbTypes.mysql:
                                            case dbTypes.mariadb:
                                                executeMySQLCommand(data.string).then(result => {
                                                    //+2 data length, since we use the first DINT/UDINT/REAL as length
                                                    let dataLen = result.length * 2 + 2;
                                                    let dataWriteType;
                                                    let data = [];
                                                    let writeData = {
                                                        area: "DT",
                                                        count: dataLen,
                                                        start: writeStringAddr,
                                                    }

                                                    switch (control) {
                                                        case modes.selectDataDINT:
                                                            writeData.type = "DINT";
                                                            dataWriteType = pew.Constants.writeDataTypes._32bit;
                                                            break;

                                                        case modes.selectDataUDINT:
                                                            writeData.type = "UDINT";
                                                            dataWriteType = pew.Constants.writeDataTypes._32bit;
                                                            break;

                                                        case modes.selectDataREAL:
                                                            writeData.type = "REAL";
                                                            dataWriteType = pew.Constants.writeDataTypes._32bitReal;
                                                            break;
                                                    }

                                                    //Push the data length first to the first register
                                                    data.push(dataLen);

                                                    if (result.length && result.length > 0) {
                                                        for (let r of result) {
                                                            if (Object.values(r)[0] === null) {
                                                                dataLen -= 2;
                                                            }
                                                            else {
                                                                data.push(Object.values(r)[0]);
                                                            }
                                                        }
                                                    }
                                                    //Since the data length could have changed, update it
                                                    data[0] = dataLen;
                                                    writeData.count = dataLen / 2;
                                                    writeData.value = data;
                                                    pew.writeMultipleRegisters(protocol, usePort, writeData, dataWriteType)
                                                        .then(() => {
                                                            setPLCStatus(STATUS.OK).catch({}).finally(() => {
                                                                done();
                                                            });
                                                        })
                                                        .catch(err => {
                                                            let errorMessage;
                                                            errorMessage = err.err_msg ? err.err_msg : err;

                                                            pew.sysLogMessage(moduleName, errorMessage);
                                                            setPLCText(cutStringLen(errorMessage, pew.Constants.MAX_SELECT_VALUES), writeStringAddr).catch(() => { }).finally(() => {
                                                                setPLCStatus(STATUS.generalErr).catch({}).finally(() => {
                                                                    done();
                                                                });
                                                            })
                                                        });
                                                }).catch(err => {
                                                    setPLCText(cutStringLen(err, stringLenMaxChars), writeStringAddr).catch({}).finally(() => {
                                                        setPLCStatus(STATUS.generalErr).catch({}).finally(() => {
                                                            done();
                                                        });
                                                    });
                                                })
                                                break;

                                            case dbTypes.postgres:
                                                executePostgresCommand(data.string).then(result => {
                                                    //+2 data length, since we use the first DINT/UDINT/REAL as length
                                                    let dataLen = result.length * 2 + 2;
                                                    let dataWriteType;
                                                    let data = [];
                                                    let writeData = {
                                                        area: "DT",
                                                        count: dataLen,
                                                        start: writeStringAddr,
                                                    }

                                                    switch (control) {
                                                        case modes.selectDataDINT:
                                                            writeData.type = "DINT";
                                                            dataWriteType = pew.Constants.writeDataTypes._32bit;
                                                            break;

                                                        case modes.selectDataUDINT:
                                                            writeData.type = "UDINT";
                                                            dataWriteType = pew.Constants.writeDataTypes._32bit;
                                                            break;

                                                        case modes.selectDataREAL:
                                                            writeData.type = "REAL";
                                                            dataWriteType = pew.Constants.writeDataTypes._32bitReal;
                                                            break;
                                                    }

                                                    //Push the data length first to the first register
                                                    data.push(dataLen);

                                                    if (result.length && result.length > 0) {
                                                        for (let r of result) {
                                                            if (Object.values(r)[0] === null) {
                                                                dataLen -= 2;
                                                            }
                                                            else {
                                                                data.push(Object.values(r)[0]);
                                                            }
                                                        }
                                                    }
                                                    //Since the data length could have changed, update it
                                                    data[0] = dataLen;
                                                    writeData.count = dataLen / 2;
                                                    writeData.value = data;
                                                    pew.writeMultipleRegisters(protocol, usePort, writeData, dataWriteType)
                                                        .then(() => {
                                                            setPLCStatus(STATUS.OK).catch({}).finally(() => {
                                                                done();
                                                            });
                                                        })
                                                        .catch(err => {
                                                            let errorMessage;
                                                            errorMessage = err.err_msg ? err.err_msg : err;

                                                            pew.sysLogMessage(moduleName, errorMessage);
                                                            setPLCText(cutStringLen(errorMessage, pew.Constants.MAX_SELECT_VALUES), writeStringAddr).catch(() => { }).finally(() => {
                                                                setPLCStatus(STATUS.generalErr).catch({}).finally(() => {
                                                                    done();
                                                                });
                                                            })
                                                        });
                                                }).catch(err => {
                                                    setPLCText(cutStringLen(err, stringLenMaxChars), writeStringAddr).catch({}).finally(() => {
                                                        setPLCStatus(STATUS.generalErr).catch({}).finally(() => {
                                                            done();
                                                        });
                                                    });
                                                })
                                                break;
                                        }
                                    }
                                }
                            }).catch(err => {
                                pew.sysLogMessage(moduleName, err);
                                setPLCText(cutStringLen(err, stringLenMaxChars)).catch({}).finally(() => {
                                    setPLCStatus(STATUS.generalErr).catch({}).finally(() => {
                                        done();
                                    });
                                });
                            })
                            break;


                        default:
                            done();
                    }
                }
                else {
                    pew.sysLogMessage(moduleName, data.err_msg);
                    done();
                }
            }
            catch (ex) {
                pew.sysLogMessage(moduleName, ex.message);
                done();
            }
        }).catch(err => {
            pew.sysLogMessage(err);
            done();
        })
    }
    //Insert data without PLC
    else {
        let insertCommand = "";
        let writeValues = [];
        //read all values
        try {
            if (sqlConfig.data.datalist.length > 0) {
                //find out the interface to use to read the data             
                for (let val of sqlConfig.data.datalist) {
                    let item = {
                        area: val.area,
                        type: val.type,
                        count: 1,
                        start: parseInt(val.address)
                    }

                    protocol.StartAddr = parseInt(val.address);

                    let result;
                    let type = val.type.toUpperCase();
                    if (type === pew.Constants.datatype.string.name) {
                        result = await pew.readString(protocol, usePort, item);
                    }
                    else {
                        let isDouble = type === pew.Constants.datatype.dint.name || type === pew.Constants.datatype.udint.name || type === pew.Constants.datatype.real.name || type === "DWORD";
                        result = await pew.readMultipleRegisters(protocol, usePort, item, isDouble);
                    }

                    if (!result.err) {
                        if (type !== pew.Constants.datatype.string.name) {
                            writeValues.push({
                                column: val.name,
                                value: result[type.toLowerCase()][0],
                                type
                            })
                        }
                        else {
                            writeValues.push({
                                column: val.name,
                                value: result[type.toLowerCase()],
                                type
                            })
                        }
                    }
                }

                let columns = "(";
                let values = "(";
                insertCommand = `insert into ${sqlConfig.data.tablename} `;
                for (let i = 0; i < writeValues.length; i++) {
                    if (i !== writeValues.length - 1) {
                        columns += writeValues[i].column + ",";
                        if (writeValues[i].type === pew.Constants.datatype.string.name) {
                            values += `'${writeValues[i].value}',`;
                        }
                        else {
                            values += writeValues[i].value + ",";
                        }
                    }
                    else {
                        columns += writeValues[i].column + ")";
                        if (writeValues[i].type === pew.Constants.datatype.string.name) {
                            values += `'${writeValues[i].value}')`;
                        }
                        else {
                            values += writeValues[i].value + ")";
                        }
                    }
                }

                insertCommand += `${columns} values ${values}`;
                switch (sqlConfig.data.db_type) {
                    case dbTypes.mssql:
                    case dbTypes.azure:
                        await executeMSSQLCommand(insertCommand);
                        break;

                    case dbTypes.mysql:
                    case dbTypes.mariadb:
                        await executeMySQLCommand(insertCommand);
                        break;

                    case dbTypes.postgres:
                        await executePostgresCommand(insertCommand);
                        break;
                }
            }
        }
        catch (ex) {
            pew.sysLogMessage(moduleName, ex.message ? ex.message : ex);
        }
        finally {
            setTimeout(plcPolling, sqlConfig.data.plc_poll * 1000);
        }
    }
}

function done() {
    setTimeout(plcPolling, sqlConfig.data.plc_poll * 1000);
}

function setPLCStatus(statusValues) {
    return new Promise((res, rej) => {
        let setStatus = {
            area: "DT",
            type: "DINT",
            count: 1,
            start: sqlConfig.data.plc_control_dt,
            value: statusValues
        }
        protocol.StartAddr = setStatus.start;
        //Service 5 is Ethernet. In case of ethernet interface, don't use the station number,
        //instead use EE
        if (thread[0].service !== 5) {
            protocol.Station = interfaceConfig.data.address_plc;
        }

        pew.writeMultipleRegisters(protocol, usePort, setStatus, pew.Constants.writeDataTypes._32bit)
            .then(() => {
                res();
            })
            .catch(err => {
                pew.sysLogMessage(moduleName, err);
                rej(err);
            });
    })
    
}

function setPLCText(errText, startAddr) {
    return new Promise((res, rej) => {
        let writeItem = {
            area: "DT",
            type: "INT",
            start: startAddr + 1,
            value: errText
        }

        let data = pew.getWriteStringRegisters(writeItem);        
        pew.writeMultipleRegisters(protocol, usePort, data).then(() => {
            res();
        }).catch(err => {
            pew.sysLogMessage(moduleName, err.err_msg);
            rej();
        });
    })    
}

async function executeMSSQLCommand(command) {
    return new Promise((res, rej) => {
        try {
            db.connect(client).then(pool => {
                pool
                    .query(command)
                    .then(result => {
                        res(result.recordset ? result.recordset : result.recordsets);
                    })
                    .catch((err) => {
                        rej(err.message);
                    });
            });
        }
        catch (ex) {
            rej(ex.message);
        }
    })
}

async function executeMySQLCommand(command) {
    return new Promise((res, rej) => {
        try {
            client.getConnection((err, connection) => {
                if (err) {
                    return rej(err.message);
                }

                connection.query(command, (err, results) => {
                    connection.release();
                    if (err) {
                        return rej(err.message);
                    }
                    //Check results.length if data has been read. if its undefined then it was probably an insert
                    res(results);
                });
            });
        }
        catch (ex) {
            rej(ex.message);
        }
    })
}

async function executePostgresCommand(command) {
    return new Promise((res, rej) => {
        client.connect().then((connection) => {
            return connection
                .query(command)
                .then((result) => {
                    res(result.rows);
                    for (let poolConnection of client._clients) {
                        poolConnection.release();
                    }                
                })
                .catch((err) => {
                    rej(err.message);
                    for (let poolConnection in client._clients) {
                        poolConnection.release();
                    }
                });
        }).catch(err => {
            rej(err.message);
        });
    })
}

function cutStringLen(actualString, maxLen) {
    if (actualString.length > maxLen) {
        return actualString.substr(0, maxLen);
    }
    else {
        return actualString;
    }
}