﻿const { MongoClient, ServerApiVersion } = require("mongodb");
const { InfluxDB, Point } = require('@influxdata/influxdb-client')
const pewModule = require('./pew_module');
const pew = new pewModule.pewModule();
const tls = require("tls");
var Mew = require('./mewtocol');
var Mew7 = require('./mewtocol7');
var Modbus = require('./modbus');
const fs = require("fs");
const path = require('path');
const moduleName = "no-sqlClient";
var nosqlClientRunning = false;
var client, db;

//Catch exceptions
process.on('uncaughtException', function (err) {
    pew.sysLogMessage(moduleName, "Uncaught exception: " + err.stack);
    disconnect();
    if (nosqlClientRunning) {
        done();
    }
})

//Read Configurations
var interfaceConfig = pew.getConfFileSync(pew.Constants.requests.READ_INTERFACE);
var portConfig = pew.getConfFileSync(pew.Constants.requests.READ_PORTS);
var nosqlConfig = pew.getConfFileSync(pew.Constants.requests.READ_NOSQL);

//If any configuration could not be loaded -> log the message and stop execution
if (interfaceConfig.err) {
    pew.sysLogMessage(moduleName, interfaceConfig.err_msg);
    return;
}

if (portConfig.err) {
    pew.sysLogMessage(moduleName, portConfig.err_msg);
    return;
}

if (nosqlConfig.err) {
    pew.sysLogMessage(moduleName, nosqlConfig.err_msg);
    return;
}

//Check configurations
var usePort, useProtocol, serviceID, protocol;
serviceID = nosqlConfig.data.insertWithoutPLC ? nosqlConfig.data.interface : interfaceConfig.data.interface_plc;
let iface = interfaceConfig.data.interface.filter(val => {
    return val.service == serviceID;
})

if (iface.length <= 0 || !iface[0].enabled) {
    return pew.sysLogMessage(moduleName, pew.Constants.COM_NOT_ENABLED);
}

useProtocol = iface[0].protocol;
//Set the protocol module
switch (useProtocol) {
    case pew.Constants.PROTOCOLS.mewtocol:
        protocol = new Mew.Mewtocol();
        break;

    case pew.Constants.PROTOCOLS.mewtocol7:
        protocol = new Mew7.Mewtocol7();
        break;

    case pew.Constants.PROTOCOLS.modbus:
        protocol = new Modbus.Modbus();
        break;

    default:
        protocol = new Mew.Mewtocol();
}

var thread = portConfig.data.thread.filter(val => {
    return val.service === serviceID;
});

if (thread.length <= 0) {
    return pew.sysLogMessage(moduleName, pew.Constants.TCP_PORT_DISABLED);
}
usePort = thread[0].port;

//Configuration all OK -> continue
let STATUS = {
    OK: 0,
    generalErr: -1,
    connectTimeout: -2
}

let dbTypes = {
    mongo: 0,
    influx: 1    
}

setupDBConnection();

function setupDBConnection() {    
    pew.getKeyContent().then(key => {
        pew.Constants.pwkey = key;
        switch (nosqlConfig.data.db_type) {
            case dbTypes.mongo:
                try {
                    let uri = nosqlConfig.data.server;
                    if (uri.search("<username>") !== -1) {
                        uri = uri.replace("<username>", nosqlConfig.data.user);
                    }
                    if (uri.search("<password>") !== -1) {
                        uri = uri.replace("<password>", pew.decryptPW(nosqlConfig.data.pass));
                    }

                    //Check if authentication via Cert
                    if (uri.search("MONGODB-X509") !== -1) {
                        const credentials = path.join(__dirname, pew.Constants.configSubfolders.nosql, nosqlConfig.data.client);
                        client = new MongoClient(uri, {
                            tlsCertificateKeyFile: credentials,
                            serverApi: ServerApiVersion.v1
                        });
                    }
                    else {
                        //in case user certificates are used                    
                        if (nosqlConfig.data.ca !== "" && nosqlConfig.data.client !== "" && nosqlConfig.data.key !== "") {
                            const secureContext = tls.createSecureContext({
                                ca: fs.readFileSync(path.join(__dirname, pew.Constants.configSubfolders.nosql, nosqlConfig.data.ca)),
                                cert: fs.readFileSync(path.join(__dirname, pew.Constants.configSubfolders.nosql, nosqlConfig.data.client)),
                                key: fs.readFileSync(path.join(__dirname, pew.Constants.configSubfolders.nosql, nosqlConfig.data.key)),
                            });
                            client = new MongoClient(uri, {
                                serverApi: {
                                    version: ServerApiVersion.v1,
                                    strict: true,
                                    deprecationErrors: true,
                                },
                                tls: nosqlConfig.data.use_encryption === 1,
                                secureContext,
                                tlsInsecure: nosqlConfig.data.ca_verification === 0,
                            });
                        }
                        else {
                            //No user defined certificates
                            client = new MongoClient(uri, {
                                serverApi: {
                                    version: ServerApiVersion.v1,
                                    strict: true,
                                    deprecationErrors: true,
                                },
                                tls: nosqlConfig.data.use_encryption === 1,
                                tlsInsecure: nosqlConfig.data.ca_verification === 0,
                            });
                        }
                    }
                    
                }
                catch (ex) {
                    setPLCStatus(STATUS.generalErr);
                    setPLCText(ex.message, nosqlConfig.data.plc_control_dt + 4);
                    pew.sysLogMessage(moduleName, ex);
                    return;
                }
                break;

            case dbTypes.influx:
                try {
                    let token = pew.decryptPW(nosqlConfig.data.pass);
                    let url = nosqlConfig.data.server;
                    client = new InfluxDB({
                        url, token, transportOptions: {
                            rejectUnauthorized: nosqlConfig.data.ca_verification === 1
                        }
                    });
                }
                catch (ex) {
                    setPLCStatus(STATUS.generalErr);
                    setPLCText(ex.message, nosqlConfig.data.plc_control_dt + 4);
                    pew.sysLogMessage(moduleName, ex);
                    return;
                }
                break;
        }

        plcPolling();
        nosqlClientRunning = true;
    }).catch(ex => {
        console.log(`${moduleName}: Failed to read key for noSQL Client: ${ex}\n`);
        setTimeout(setupDBConnection, 5000);
    })
}

async function plcPolling() {

    if (!nosqlConfig.data.insertWithoutPLC) {
        let readItem = {
            area: "DT",
            type: "DINT",
            count: 5,
            start: nosqlConfig.data.plc_control_dt
        }
        const modes = {
            insert: 1, //insert data
            reserved: 2, //for compatibility not used
            find: 3 // find entries
        }
        let maxWriteStringLen = pew.Constants.MAX_STRING_LEN_REQUEST;
        protocol.StartAddr = readItem.start;
        try {
            let data = await pew.readMultipleRegisters(protocol, usePort, readItem, true);
            if (!data.err) {
                let control = data.dint[0];
                let item = {
                    area: "DT",
                    start: data.dint[1]
                }
                let writeStringAddr = data.dint[2];
                let maxlen;

                switch (control) {
                    //insert data (single or multiple)
                    case modes.insert:
                        //Read the max. writable string length of PLC                        
                        maxlen = await getMaxStringSize(data.dint[2]);
                        if (maxlen !== -1) {
                            maxWriteStringLen = maxlen;
                        }
                        try {
                            let payload = await pew.readString(protocol, usePort, item);
                            if (!payload.err) {
                                if (payload.string === pew.Constants.EMPTY_STRING) {
                                    setPLCText(cutStringLen(pew.Constants.EMPTY_STRING_NOT_ALLOWED, maxWriteStringLen), writeStringAddr);
                                    setPLCStatus(STATUS.generalErr);
                                    done();
                                }
                                else {
                                    switch (nosqlConfig.data.db_type) {
                                        case dbTypes.mongo:
                                            //Read collection string
                                            try {
                                                let collectionItem = {
                                                    area: "DT",
                                                    start: data.dint[3]
                                                }
                                                let collection = await pew.readString(protocol, usePort, collectionItem);
                                                //In case the insert data is not an array (single data), insert it anyway into an array
                                                //since we only use "insertmany"
                                                let insertJsonData = JSON.parse(payload.string);
                                                let insertData;
                                                if (Array.isArray(insertJsonData)) {
                                                    insertData = insertJsonData;
                                                    //check if in object ts (timestamp) a property of $date is available. If so
                                                    //make a new date out of it
                                                    for (let i = 0; i < insertData.length; i++) {
                                                        if (insertData[i].ts && insertData[i].ts.$date) {
                                                            insertData[i].ts = new Date(insertData[i].ts.$date);
                                                        }
                                                    }
                                                }
                                                else {
                                                    insertData = [];
                                                    //check if in object ts (timestamp) a property of $date is available. If so
                                                    //make a new date out of it
                                                    if (insertJsonData.ts && insertJsonData.ts.$date) {
                                                        insertJsonData.ts = new Date(insertJsonData.ts.$date);
                                                    }
                                                    insertData.push(insertJsonData);
                                                }
                                                //Check if connection is already established
                                                if (typeof client.topology === typeof undefined || !client.topology.isConnected()) {
                                                    await client.connect();
                                                    db = client.db(nosqlConfig.data.instance_name);
                                                }

                                                await db.collection(collection.string).insertMany(insertData);
                                                await setPLCText(cutStringLen(pew.Constants.OK, maxWriteStringLen), writeStringAddr);
                                                await setPLCStatus(STATUS.OK);
                                                done();
                                            }
                                            catch (ex) {
                                                pew.sysLogMessage(moduleName, ex);
                                                await setPLCText(cutStringLen(ex.message, maxWriteStringLen), writeStringAddr)
                                                await setPLCStatus(STATUS.generalErr);
                                                done();
                                            }
                                            break;

                                        case dbTypes.influx:
                                            try {
                                                //Read the bucket name
                                                let readItem = {
                                                    area: "DT",
                                                    start: data.dint[3]
                                                }
                                                let bucket = await pew.readString(protocol, usePort, readItem);
                                                //Now read the measurement
                                                readItem.start = data.dint[4];
                                                let measurement = await pew.readString(protocol, usePort, readItem);

                                                let contentData = JSON.parse(payload.string);
                                                if (typeof contentData.data === pew.Constants.UNDEFINED) {
                                                    setPLCText(cutStringLen(pew.Constants.DATA_PROPERTY_MISSING, maxWriteStringLen), writeStringAddr).then(() => {
                                                        setPLCStatus(STATUS.generalErr);
                                                    });                                                    
                                                    done();
                                                }
                                                else {
                                                    let points = [];
                                                    let point;
                                                    let tagname = "", tagvalue = "";
                                                    let writeClient = client.getWriteApi(nosqlConfig.data.instance_name, bucket.string, "ns");
                                                    //is an array -> multi data
                                                    if (Array.isArray(contentData.data)) {
                                                        for (let val of contentData.data) {
                                                            if (typeof val.type === pew.Constants.UNDEFINED || !pew.isValidDataType(val.type) || typeof val.name === pew.Constants.UNDEFINED || typeof val.value === pew.Constants.UNDEFINED)
                                                            {
                                                                setPLCText(cutStringLen(pew.Constants.INFLUX_PROPERTY_NOK, maxWriteStringLen), writeStringAddr).then(() => {
                                                                    setPLCStatus(STATUS.generalErr);
                                                                });                                                                
                                                                done();
                                                                return;
                                                            }
                                                            tagname = typeof val.tagName === pew.Constants.UNDEFINED ? "" : val.tagName;
                                                            tagvalue = typeof val.tagValue === pew.Constants.UNDEFINED ? "" : val.tagValue;
                                                            switch (val.type.toUpperCase()) {
                                                                case pew.Constants.datatype.int.name:
                                                                case pew.Constants.datatype.dint.name:
                                                                    point = new Point(measurement.string)
                                                                        .tag(tagname, tagvalue)
                                                                        .intField(val.name, val.value)
                                                                    break;

                                                                case pew.Constants.datatype.uint.name:
                                                                case pew.Constants.datatype.udint.name:
                                                                    point = new Point(measurement.string)
                                                                        .tag(tagname, tagvalue)
                                                                        .uintField(val.name, val.value)
                                                                    break;

                                                                case pew.Constants.datatype.real.name:
                                                                    point = new Point(measurement.string)
                                                                        .tag(tagname, tagvalue)
                                                                        .floatField(val.name, val.value)
                                                                    break;

                                                                case pew.Constants.datatype.string.name:
                                                                    point = new Point(measurement.string)
                                                                        .tag(tagname, tagvalue)
                                                                        .stringField(val.name, val.value)
                                                                    break;

                                                                case pew.Constants.datatype.bool.name:
                                                                    point = new Point(measurement.string)
                                                                        .tag(tagname, tagvalue)
                                                                        .booleanField(val.name, val.value)
                                                                    break;
                                                            }

                                                            if (point) {
                                                                points.push(point);
                                                            }
                                                        }

                                                        writeClient.writePoints(points);
                                                        writeClient.flush()
                                                            .then(() => {
                                                                setPLCText(pew.Constants.OK, writeStringAddr).then(() => {
                                                                    setPLCStatus(STATUS.OK);
                                                                });
                                                                done();
                                                            })
                                                            .catch((ex) => {
                                                                setPLCText(cutStringLen(ex.message, maxWriteStringLen), writeStringAddr).then(() => {
                                                                    setPLCStatus(STATUS.generalErr);
                                                                });                                                            
                                                                done();
                                                            });
                                                        writeClient.close();
                                                    }
                                                    else {
                                                        //single data
                                                        if (typeof contentData.data.type === pew.Constants.UNDEFINED || !pew.isValidDataType(contentData.data.type || typeof contentData.data.name === pew.Constants.UNDEFINED || typeof contentData.data.value === pew.Constants.UNDEFINED)) {
                                                            setPLCText(cutStringLen(pew.Constants.INFLUX_PROPERTY_NOK, maxWriteStringLen), writeStringAddr).then(() => {
                                                                setPLCStatus(STATUS.generalErr);
                                                            });                                                            
                                                            done();
                                                            return;
                                                        }
                                                        switch (contentData.data.type.toUpperCase()) {
                                                            case pew.Constants.datatype.int.name:
                                                            case pew.Constants.datatype.dint.name:
                                                                point = new Point(measurement.string)
                                                                    .tag(typeof contentData.data.tagName === pew.Constants.UNDEFINED ? "" : contentData.data.tagName, typeof contentData.data.tagValue === pew.Constants.UNDEFINED ? "" : contentData.data.tagValue)
                                                                    .intField(contentData.name, contentData.value)
                                                                break;

                                                            case pew.Constants.datatype.uint.name:
                                                            case pew.Constants.datatype.udint.name:
                                                                point = new Point(measurement.string)
                                                                    .tag(typeof contentData.data.tagName === pew.Constants.UNDEFINED ? "" : contentData.data.tagName, typeof contentData.data.tagValue === pew.Constants.UNDEFINED ? "" : contentData.data.tagValue)
                                                                    .uintField(contentData.data.name, contentData.data.value)
                                                                break;

                                                            case pew.Constants.datatype.real.name:
                                                                point = new Point(measurement.string)
                                                                    .tag(typeof contentData.data.tagName === pew.Constants.UNDEFINED ? "" : contentData.data.tagName, typeof contentData.data.tagValue === pew.Constants.UNDEFINED ? "" : contentData.data.tagValue)
                                                                    .floatField(contentData.data.name, contentData.data.value)
                                                                break;

                                                            case pew.Constants.datatype.string.name:
                                                                point = new Point(measurement.string)
                                                                    .tag(typeof contentData.data.tagName === pew.Constants.UNDEFINED ? "" : contentData.data.tagName, typeof contentData.data.tagValue === pew.Constants.UNDEFINED ? "" : contentData.data.tagValue)
                                                                    .stringField(contentData.name, contentData.value)
                                                                break;

                                                            case pew.Constants.datatype.bool.name:
                                                                point = new Point(measurement.string)
                                                                    .tag(typeof contentData.data.tagName === pew.Constants.UNDEFINED ? "" : contentData.data.tagName, typeof contentData.data.tagValue === pew.Constants.UNDEFINED ? "" : contentData.data.tagValue)
                                                                    .booleanField(contentData.data.name, contentData.data.value)
                                                                break;
                                                        }

                                                        if (point) {
                                                            points.push(point);
                                                            writeClient.writePoints(points);
                                                            writeClient.flush()
                                                                .then(() => {
                                                                    setPLCText(pew.Constants.OK, writeStringAddr).then(() => {
                                                                        setPLCStatus(STATUS.OK);
                                                                    });
                                                                    done();
                                                                })
                                                                .catch((ex) => {
                                                                    setPLCText(cutStringLen(ex.message, maxWriteStringLen), writeStringAddr).then(() => {
                                                                        setPLCStatus(STATUS.generalErr);
                                                                    });                                                                    
                                                                    done();
                                                                });
                                                            writeClient.close();
                                                        }
                                                    }
                                                }
                                            }
                                            catch (ex) {
                                                pew.sysLogMessage(moduleName, ex);
                                                await setPLCText(cutStringLen(ex.message, maxWriteStringLen), writeStringAddr)
                                                await setPLCStatus(STATUS.generalErr);
                                                done();
                                            }
                                            break;
                                    }
                                }
                            }
                        }
                        catch (err) {
                            pew.sysLogMessage(moduleName, err.err_msg);
                            setPLCText(cutStringLen(err.message, stringLenMaxChars), writeStringAddr).then(() => {
                                setPLCStatus(STATUS.generalErr);
                            })                            
                            done();
                        }
                        break;

                    //Find data
                    case modes.find:
                        //Read the max. writable string length of PLC
                        maxlen = await getMaxStringSize(data.dint[2]);
                        if (maxlen !== -1) {
                            maxWriteStringLen = maxlen;
                        }
                        try {
                            let payload = await pew.readString(protocol, usePort, item);
                            if (!payload.err) {
                                switch (nosqlConfig.data.db_type) {
                                    case dbTypes.mongo:
                                        //Read collection string
                                        try {
                                            let collectionItem = {
                                                area: "DT",
                                                start: data.dint[3]
                                            }
                                            let collection = await pew.readString(protocol, usePort, collectionItem);
                                            //get the find options from payload
                                            let options = payload.string === "" ? {} : JSON.parse(payload.string);

                                            //Check if connection is already established
                                            if (typeof client.topology === typeof undefined || !client.topology.isConnected()) {
                                                await client.connect();
                                                db = client.db(nosqlConfig.data.instance_name);
                                            }

                                            let result = await db.collection(collection.string).find(options.filter ? options.filter : {})
                                                .limit(options.limit ? options.limit : 0)
                                                .project(options.projection ? options.projection : {})
                                                .sort(options.sort ? options.sort : {})
                                                .toArray();
                                            await setPLCText(cutStringLen(JSON.stringify(result), maxWriteStringLen), writeStringAddr);
                                            await setPLCStatus(STATUS.OK);
                                            done();
                                        }
                                        catch (ex) {
                                            pew.sysLogMessage(moduleName, ex);
                                            await setPLCText(cutStringLen(ex.message, maxWriteStringLen), writeStringAddr)
                                            await setPLCStatus(STATUS.generalErr);
                                            done();
                                        }
                                        break;

                                    case dbTypes.influx:
                                        try {
                                            let queryClient = client.getQueryApi(nosqlConfig.data.instance_name);
                                            let responseObj = [];
                                            queryClient.queryRows(payload.string, {
                                                next: (row, tableMeta) => {
                                                    const tableObject = tableMeta.toObject(row);
                                                    responseObj.push(tableObject);                                                    
                                                },
                                                error: (error) => {
                                                    setPLCText(cutStringLen(error.message, maxWriteStringLen), writeStringAddr).then(() => {
                                                        setPLCStatus(STATUS.generalErr);
                                                    });
                                                    done();
                                                },
                                                complete: () => {                                                    
                                                    setPLCText(cutStringLen(JSON.stringify(responseObj), maxWriteStringLen), writeStringAddr).then(() => {
                                                        setPLCStatus(STATUS.OK);
                                                    })                                                    
                                                    done();
                                                },
                                            });
                                        }
                                        catch (ex) {
                                            pew.sysLogMessage(moduleName, ex);
                                            await setPLCText(cutStringLen(ex.message, maxWriteStringLen), writeStringAddr)
                                            await setPLCStatus(STATUS.generalErr);
                                            done();
                                        }
                                        break;
                                }                                
                            }
                        }
                        catch (err) {
                            pew.sysLogMessage(moduleName, err.err_msg);
                            setPLCText(cutStringLen(err.message, stringLenMaxChars), writeStringAddr)
                            setPLCStatus(STATUS.generalErr);
                            done();
                        }
                        break;
                    default:
                        done();
                }
            }
            else {
                pew.sysLogMessage(moduleName, data.err_msg);
                done();
            }
        }
        catch (ex) {
            pew.sysLogMessage(moduleName, ex.message);
            done();
        }
    }
    else {
        //Read collection string
        switch (nosqlConfig.data.db_type) {
            case dbTypes.mongo:                
                try {
                    if (nosqlConfig.data.datalist.length > 0) {
                        //In case the insert data is not an array (single data), insert it anyway into an array
                        //since we only use "insertmany"
                        let insertDataAsArray = [];
                        let insertData = {};

                        for (let val of nosqlConfig.data.datalist) {
                            let item = {
                                area: val.area,
                                type: val.type,
                                count: 1,
                                start: parseInt(val.address)
                            }
                            protocol.StartAddr = item.start;
                            let result;
                            let type = val.type.toUpperCase();
                            try {
                                if (type === pew.Constants.datatype.string.name) {
                                    result = await pew.readString(protocol, usePort, item);
                                }
                                else {
                                    let isDouble = type === pew.Constants.datatype.dint.name || type === pew.Constants.datatype.udint.name || type === pew.Constants.datatype.real.name || type === "DWORD";
                                    result = await pew.readMultipleRegisters(protocol, usePort, item, isDouble);
                                }
                            }
                            catch (ex) {
                                console.log(ex);
                            }

                            if (!result.err) {
                                if (type !== pew.Constants.datatype.string.name) {
                                    insertData[val.name] = result[type.toLowerCase()][0];
                                }
                                else {
                                    insertData[val.name] = result[type.toLowerCase()];
                                }
                            }
                        }

                        if (Object.keys(insertData).length !== 0) {
                            insertDataAsArray.push(insertData);
                            //Check if connection is already established
                            if (typeof client.topology === typeof undefined || !client.topology.isConnected()) {
                                await client.connect();
                                db = client.db(nosqlConfig.data.instance_name);
                            }
                            await db.collection(nosqlConfig.data.collection).insertMany(insertDataAsArray);
                        }
                        else {
                            pew.sysLogMessage(moduleName, "Data could not be read");
                        }
                        
                    }
                }
                catch (ex) {
                    pew.sysLogMessage(moduleName, ex);                    
                }
                finally {
                    done();
                }
                break;

            case dbTypes.influx:
                try {
                    if (nosqlConfig.data.datalist.length > 0) {
                        //In case the insert data is not an array (single data), insert it anyway into an array
                        //since we only use "insertmany"
                        let insertDataAsArray = [];                        
                        let point;
                        for (let val of nosqlConfig.data.datalist) {
                            let item = {
                                area: val.area,
                                type: val.type,
                                count: 1,
                                start: parseInt(val.address)
                            }
                            protocol.StartAddr = item.start;
                            let result;
                            let type = val.type.toUpperCase();
                            try {
                                if (type === pew.Constants.datatype.string.name) {
                                    result = await pew.readString(protocol, usePort, item);
                                }
                                else {
                                    let isDouble = type === pew.Constants.datatype.dint.name || type === pew.Constants.datatype.udint.name || type === pew.Constants.datatype.real.name || type === "DWORD";
                                    result = await pew.readMultipleRegisters(protocol, usePort, item, isDouble);
                                }
                            }
                            catch (ex) {
                                console.log(ex);
                            }

                            if (!result.err) {
                                switch (type) {
                                    case pew.Constants.datatype.int.name:
                                    case pew.Constants.datatype.dint.name:                                    
                                        point = new Point(nosqlConfig.data.measurement)
                                            .tag(val.tagname, val.tagvalue)
                                            .intField(val.name, result[type.toLowerCase()][0]);
                                        insertDataAsArray.push(point);
                                            break;

                                    case pew.Constants.datatype.uint.name:
                                    case pew.Constants.datatype.udint.name:
                                        point = new Point(nosqlConfig.data.measurement)
                                            .tag(val.tagname, val.tagvalue)
                                            .uintField(val.name, result[type.toLowerCase()][0]);
                                        insertDataAsArray.push(point);
                                        break;

                                    case pew.Constants.datatype.real.name:
                                        point = new Point(nosqlConfig.data.measurement)
                                            .tag(val.tagname, val.tagvalue)
                                            .floatField(val.name, result[type.toLowerCase()][0]);
                                        insertDataAsArray.push(point);
                                        break;

                                    case pew.Constants.datatype.string.name:
                                        point = new Point(nosqlConfig.data.measurement)
                                            .tag(val.tagname, val.tagvalue)
                                            .stringField(val.name, result[type.toLowerCase()]);
                                        insertDataAsArray.push(point);
                                        break;

                                    case pew.Constants.datatype.bool.name:
                                        point = new Point(nosqlConfig.data.measurement)
                                            .tag(val.tagname, val.tagvalue)
                                            .booleanField(val.name, result[type.toLowerCase()][0]);
                                        insertDataAsArray.push(point);
                                        break;
                                }
                                
                            }
                        }

                        if (insertDataAsArray.length !== 0) {
                            let writeClient = client.getWriteApi(nosqlConfig.data.instance_name, nosqlConfig.data.collection, 'ns')
                            writeClient.writePoints(insertDataAsArray);
                            writeClient.flush(true).catch(ex => {
                                pew.sysLogMessage(moduleName, ex.message);
                            })
                            writeClient.close();
                        }
                        else {
                            pew.sysLogMessage(moduleName, "Data could not be read");
                        }

                    }
                }
                catch (ex) {
                    pew.sysLogMessage(moduleName, ex);
                }
                finally {
                    done();
                }
                break;
        }        
    }
    
}

async function setPLCStatus(statusValues) {
    let setStatus = {
        area: "DT",
        type: "DINT",
        count: 1,
        start: nosqlConfig.data.plc_control_dt,
        value: statusValues
    }
    protocol.StartAddr = setStatus.start;
    //Service 5 is Ethernet. In case of ethernet interface, don't use the station number,
    //instead use EE
    if (thread[0].service !== 5) {
        protocol.Station = interfaceConfig.data.address_plc;
    }
    try {
        await pew.writeMultipleRegisters(protocol, usePort, setStatus, pew.Constants.writeDataTypes._32bit);
    }
    catch (err)
    {
        pew.sysLogMessage(moduleName, err);
        rej(err);
    }
}

async function setPLCText(errText, startAddr) {
    try {
        let maxStringLen = await getMaxStringSize(startAddr);
        let errTextCut;
        if (maxStringLen != -1) {
            errTextCut = cutStringLen(errText, maxStringLen);
        }
        else {
            errTextCut = errText;
        }

        let writeItem = {
            area: "DT",
            type: "INT",
            start: startAddr + 1,
            value: errTextCut
        }

        let data = pew.getWriteStringRegisters(writeItem);
        await pew.writeMultipleRegisters(protocol, usePort, data);
    }
    catch (ex) {
        pew.sysLogMessage(moduleName, ex.message);
    }
}


async function getMaxStringSize(addrOfString) {
    let readItem = {
        area: "DT",
        type: "INT",
        count: 1,
        start: addrOfString
    }

    try {
        protocol.StartAddr = addrOfString;
        let result = await pew.readMultipleRegisters(protocol, usePort, readItem);
        if (!result.err) {
            return result.int[0];
        }
        return -1;
    }
    catch (err) {
        pew.sysLogMessage(moduleName, err);
        return -1;
    }
}

function cutStringLen(actualString, maxLen) {
    if (actualString.length > maxLen) {
        return actualString.substr(0, maxLen);
    }
    else {
        return actualString;
    }
}

function disconnect() {
    try {
        if (typeof client.topology !== typeof undefined) {
            if (client.topology.isConnected()) {
                client.close().catch(err => pew.sysLogMessage(err));
            }
        }
    }
    catch (ex) {}
}

function done() {
    setTimeout(plcPolling, nosqlConfig.data.plc_poll * 1000);
}