"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _numericn = _interopRequireDefault(require("./numericn"));

var _writableTrackingBuffer = _interopRequireDefault(require("../tracking-buffer/writable-tracking-buffer"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const NULL_LENGTH = Buffer.from([0x00]);
const Numeric = {
  id: 0x3F,
  type: 'NUMERIC',
  name: 'Numeric',
  declaration: function (parameter) {
    return 'numeric(' + this.resolvePrecision(parameter) + ', ' + this.resolveScale(parameter) + ')';
  },
  resolvePrecision: function (parameter) {
    if (parameter.precision != null) {
      return parameter.precision;
    } else if (parameter.value === null) {
      return 1;
    } else {
      return 18;
    }
  },
  resolveScale: function (parameter) {
    if (parameter.scale != null) {
      return parameter.scale;
    } else {
      return 0;
    }
  },

  generateTypeInfo(parameter) {
    let precision;

    if (parameter.precision <= 9) {
      precision = 0x05;
    } else if (parameter.precision <= 19) {
      precision = 0x09;
    } else if (parameter.precision <= 28) {
      precision = 0x0D;
    } else {
      precision = 0x11;
    }

    return Buffer.from([_numericn.default.id, precision, parameter.precision, parameter.scale]);
  },

  generateParameterLength(parameter, options) {
    if (parameter.value == null) {
      return NULL_LENGTH;
    }

    const precision = parameter.precision;

    if (precision <= 9) {
      return Buffer.from([0x05]);
    } else if (precision <= 19) {
      return Buffer.from([0x09]);
    } else if (precision <= 28) {
      return Buffer.from([0x0D]);
    } else {
      return Buffer.from([0x11]);
    }
  },

  *generateParameterData(parameter, options) {
    if (parameter.value == null) {
      return;
    }

    const sign = parameter.value < 0 ? 0 : 1;
    const value = Math.round(Math.abs(parameter.value * Math.pow(10, parameter.scale)));

    if (parameter.precision <= 9) {
      const buffer = Buffer.alloc(5);
      buffer.writeUInt8(sign, 0);
      buffer.writeUInt32LE(value, 1);
      yield buffer;
    } else if (parameter.precision <= 19) {
      const buffer = new _writableTrackingBuffer.default(10);
      buffer.writeUInt8(sign);
      buffer.writeUInt64LE(value);
      yield buffer.data;
    } else if (parameter.precision <= 28) {
      const buffer = new _writableTrackingBuffer.default(14);
      buffer.writeUInt8(sign);
      buffer.writeUInt64LE(value);
      buffer.writeUInt32LE(0x00000000);
      yield buffer.data;
    } else {
      const buffer = new _writableTrackingBuffer.default(18);
      buffer.writeUInt8(sign);
      buffer.writeUInt64LE(value);
      buffer.writeUInt32LE(0x00000000);
      buffer.writeUInt32LE(0x00000000);
      yield buffer.data;
    }
  },

  validate: function (value) {
    if (value == null) {
      return null;
    }

    value = parseFloat(value);

    if (isNaN(value)) {
      throw new TypeError('Invalid number.');
    }

    return value;
  }
};
var _default = Numeric;
exports.default = _default;
module.exports = Numeric;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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