#!/bin/bash

DATA_DIR="/tmp/.runc_nsnet"
BASE_NET="172.27.0.0/28"
CONF_PATH="/etc/jmlauncher/container-network.env"

[ -e "$CONF_PATH" ] && . "$CONF_PATH"

##### functions start #####
# our map data
declare -A DATA_MAP

# split ip address
# $1 -> ip address to split
# ret retval -> array with splitted ip
split_ip_address() {
  OIFS=$IFS
  IFS='.'
  retval=($1)
  IFS=$OIFS
  if [ "${#retval[@]}" -ne 4 ]; then
    return 1
  fi
  # check mask
  OIFS=$IFS
  IFS='/'
  laste=(${retval[3]})
  IFS=$OIFS
  if [ "${#laste[@]}" -eq 2 ]; then
    retval[3]="${laste[0]}"
    retval[4]="${laste[1]}"
  fi
  return 0
}

# load stored networks data into our map
load_map_data() {
  for f in "$DATA_DIR"/*; do
    [ -e "$f" ] || continue
    fname="$(basename -- $f)"
    DATA_MAP["$fname"]="$(cat $f)"
  done
}

# $1 -> uuid
# retvat -> size of elems
get_map_elem_length() {
  OIFS=$IFS
  IFS=';'
  retval=(${DATA_MAP[$1]})
  IFS=$OIFS
  retval="${#retval[@]}"
}

# $1 -> uuid
# $2 -> index of the value
# retvat -> value or not set
get_map_elem() {
  OIFS=$IFS
  IFS=';'
  retval=(${DATA_MAP[$1]})
  IFS=$OIFS
  if [ "$2" -ge "${#retval[@]}" ]; then
    unset retval
  fi
  retval="${retval[$2]}"
}

# store internal data
# $1 -> uuid
# $2 -> data to save
save_data() {
  printf "$2" > "$DATA_DIR/$1"
}

# store internal data
# $1 -> uuid
# $2 -> data to concat
cat_data() {
  printf ";$2" >> "$DATA_DIR/$1"
}

# delete data
# $1 -> uuid
delete_data() {
  rm "$DATA_DIR/$1"
}

# create uuid
new_uuid() {
  while : ; do
    newuuid=$(tr -dc 'a-f0-9' < /dev/urandom | head -c8)
    if [ ! -v DATA_MAP["$newuuid"] ]; then
      echo "$newuuid"
      break
    fi
  done
}

# get an ip address to setup the network
# the first time it will use BASE_NET, subsequet call
# will find a free subnet
# retval -> ip address
get_ip_net_address() {
  split_ip_address "$BASE_NET"
  if [ "${#retval[@]}" -lt 4 ] || [ "${#retval[@]}" -gt 5 ]; then
    echo 'Error, the subnet provided is not valid'
    exit 1
  elif [ "${#retval[@]}" -eq 5 ]; then
    net_mask="${retval[4]}"
  else
    echo 'Error, please provide a network mask'
    exit 1
  fi
  # check net mask
  if [ "$net_mask" -lt 8 ] || [ "$net_mask" -gt 30 ]; then
    echo 'Error, network mask not valid'
    exit 1
  fi

  nv=$(( (retval[0]<<24)+(retval[1]<<16)+(retval[2]<<8)+(retval[3]) ))

  if [ "${#DATA_MAP[@]}" -gt 0 ]; then
    # fill used nets
    allnetval=()
    for k in "${!DATA_MAP[@]}"; do 
      get_map_elem "$k" 0
      split_ip_address "$retval"
      [ "${#retval[@]}" -ne 5 ] && continue
      allnetval["${#allnetval[@]}"]=$(( (retval[0]<<24)+(retval[1]<<16)+(retval[2]<<8)+(retval[3]) ))
    done

    # find first free subnet
    nvmask=$(( ((2**retval[4])-1)<<(32-retval[4]) ))
    nv=$(( nv&nvmask ))
    while [[ " ${allnetval[@]} " =~ " ${nv} " ]]; do
      nv=$(( nv + (1<<(32-retval[4])) ))
      if [ $(( nv>>24 )) -eq 255 ]; then
        echo 'Error, unable to find a new subnet'
        exit 1
      fi
    done
  fi

  # check subnet
  nvmask=$(( ((2**retval[4])-1)<<(32-retval[4]) ))
  nv=$(( nv&nvmask ))
  # return ip
  retval[0]=$(( nv>>24 ))
  retval[1]=$(( (nv>>16)&255 ))
  retval[2]=$(( (nv>>8)&255 ))
  retval[3]=$(( (nv)&255 ))
}

# $1 -> host ip
# $2 -> host port
# $3 -> container port
# $4 -> protocol (default tcp)
# retval -> socat pid
publish_port() {
  if [ -z "$2" ] || [ -z "$3" ]; then
    return
  fi
  pb_host_ip="$1"
  if [ -z "$pb_host_ip" ] || [ "$pb_host_ip" == 'null' ]; then
    pb_host_ip='0.0.0.0'
  fi
  pb_host="$2"
  pb_container="$3"
  pb_protocol="$4"
  if [ -z "$pb_protocol" ] || ([ "$pb_protocol" != 'tcp' ] && [ "$pb_protocol" != 'udp' ]); then
    pb_protocol='tcp'
  fi

  echo "Publish port $pb_host_ip:$pb_host:$pb_container/$pb_protocol"
  socat "$pb_protocol"-listen:"$pb_host",bind="$pb_host_ip",reuseaddr,fork "$pb_protocol"-connect:"$machine_ip":"$pb_container" &
  retval=$!
  # do not setup iptables port forwarding, use socat
  return

  # setup local port forwarding
  iptables -t nat \
    -A OUTPUT \
    --match addrtype --dst-type LOCAL \
    -p "$pb_protocol" -m "$pb_protocol" --dport "$pb_host" \
    -j DNAT \
    --to-destination "$machine_ip":"$pb_container"
  # setup external port forwarding
  iptables -t nat \
    -A PREROUTING \
    --match addrtype --dst-type LOCAL \
    -p "$pb_protocol" -m "$pb_protocol" --dport "$pb_host" \
    -j DNAT \
    --to-destination "$machine_ip":"$pb_container"
}
##### functions end #####

# check root
if [ "$(id -u)" -ne 0 ]; then
  echo 'Error, please run as root'
  exit 1
fi

# check number of parameters
if [ "$#" -le 1 ] ||
  ([ "$1" != 'create' ] && [ "$1" != 'remove' ]) ||
  ([ "$1" == 'create' ] && [ "$#" -ne 3 ]) || ([ "$1" == 'remove' ] && [ "$#" -ne 2 ]); then
  echo 'Usage:'
  echo './container-network create|remove unique_name [path_of_configjson]'
  exit 1
fi

# create data folder
[ ! -d "$DATA_DIR" ] && mkdir -p "$DATA_DIR"

# load our data
load_map_data

# use unique_name as our uuid (truncated at 10 characters)
uuid="$(echo "$2" | cut -c 1-10)"

# remove network
if [ "$1" == 'remove' ]; then
  if [ ! -v DATA_MAP["$uuid"] ]; then
    echo 'Network not found'
    exit 0
  fi

  # remove network interfaces
  ip netns exec ns_"$uuid" ip link set lo down
  ip link set veth"$uuid" down
  ip link set br_"$uuid" down
  ip link delete veth"$uuid"
  ip link delete br_"$uuid"
  ip netns delete ns_"$uuid"
  # remove rules
  get_map_elem "$uuid" 0
  net_ip_mask="$retval"
  iptables -D RUNC-ISOLATION-ONE -i br_"$uuid" ! -o br_"$uuid" -j RUNC-ISOLATION-TWO
  iptables -D RUNC-ISOLATION-TWO -o br_"$uuid" -j DROP
  iptables -t nat -D POSTROUTING -s "$net_ip_mask" ! -o br_"$uuid" -j MASQUERADE
  # remove iptables mappings
  get_map_elem "$uuid" 2
  rows=$(iptables -S -t nat | grep "to-destination $retval")
  if [ ! -z "$rows" ]; then
    while IFS= read -r line; do
      delcmd="${line/-A/-D}"
      iptables -t nat $delcmd
    done <<< "$rows"
  fi
  # remove socat mappings
  get_map_elem_length "$uuid"
  plength="$retval"
  # i=3, skip saved ip addresses
  for (( i=3; i<$((plength)); i++ )); do
    get_map_elem "$uuid" "$i"
    kill -9 "$retval"
  done
  # delete data
  delete_data "$uuid"
  exit 0
fi

# create network
PATH_DIR="$3"
PATH_CONFIG_JSON="$3/config.json"
PATH_PORTS_JSON="$3/ports.json"

# check presence of config.json
if [ ! -f "$PATH_CONFIG_JSON" ]; then
  echo "$PATH_DIR must contain the container config.json"
  exit 1
fi

# find out uuid from config.json
uuid_js=$(jq -r '.linux.namespaces[] | select(.type=="network") | .path' "$PATH_CONFIG_JSON" | sed -n 's/^.*ns_//p')
if [ -z "$uuid_js" ]; then
  echo 'No network namespace specified: use host networking'
  exit 0
elif [ "$uuid" != "$uuid_js" ]; then
  echo "Error, network $uuid_js and unique_name $uuid are not the same, but they should"
  exit 1
fi

# previously created network?
if [ -v DATA_MAP["$uuid"] ]; then
  echo 'Network already created'
  exit 0
fi

# create base ip address
get_ip_net_address

# setup our net addresses
net_mask="${retval[4]}"
net_ip="${retval[0]}.${retval[1]}.${retval[2]}.${retval[3]}"
bridge_ip="${retval[0]}.${retval[1]}.${retval[2]}.$((retval[3]+1))"
machine_ip="${retval[0]}.${retval[1]}.${retval[2]}.$((retval[3]+2))"
# save
save_data "$uuid" "$net_ip/$net_mask;$bridge_ip;$machine_ip"

echo "Setup network $net_ip/$net_mask, gateway $bridge_ip, container $machine_ip"
# enable forwarding
echo 1 > /proc/sys/net/ipv4/ip_forward
# create a linux bridge 'br_*'
ip link add name br_"$uuid" type bridge
# create pair of virtual eth
ip link add dev vetht"$uuid" type veth peer name veth"$uuid"
# create a network namespace 'ns_*'
ip netns add ns_"$uuid"
# move 'vetht*' to the network namespace 'ns_*'
ip link set vetht"$uuid" netns ns_"$uuid"
# join 'veth*' network interface to 'br_*' bridge
ip link set veth"$uuid" master br_"$uuid"
# * assign an ipv4 address to 'br_*'
ip addr add "$bridge_ip/$net_mask" brd + dev br_"$uuid"
# assign an ipv4 address to 'vetht*'
ip netns exec ns_"$uuid" ip addr add "$machine_ip/$net_mask" dev vetht"$uuid"
# rename 'vetht*' device to eth0 and up it
ip netns exec ns_"$uuid" ip link set vetht"$uuid" name eth0
ip netns exec ns_"$uuid" ip link set eth0 up
# up network namespace loop device
ip netns exec ns_"$uuid" ip link set lo up
# up 'veth*' device
ip link set veth"$uuid" up
# up 'br_*' device
ip link set br_"$uuid" up
# add bridge 'br_*' as default gateway for the container network
ip netns exec ns_"$uuid" ip route add default via "$bridge_ip"

# setup isolation rules
if [ ! -f "$DATA_DIR/.setuprules" ]; then
  iptables -N RUNC-ISOLATION-ONE
  iptables -A RUNC-ISOLATION-ONE -j RETURN
  iptables -N RUNC-ISOLATION-TWO
  iptables -A RUNC-ISOLATION-TWO -j RETURN
  iptables -I FORWARD 1 -j RUNC-ISOLATION-ONE
  echo 1 > "$DATA_DIR/.setuprules"
fi
iptables -I RUNC-ISOLATION-ONE 1 -i br_"$uuid" ! -o br_"$uuid" -j RUNC-ISOLATION-TWO
iptables -I RUNC-ISOLATION-TWO 1 -o br_"$uuid" -j DROP

# add connectivity
iptables -t nat -A POSTROUTING -s "$net_ip/$net_mask" ! -o br_"$uuid" -j MASQUERADE

# publish ports
if [ -f "$PATH_PORTS_JSON" ]; then
  count="$(jq '.mappings | length' "$PATH_PORTS_JSON")"
  count=$(( count - 1 ))
  for i in $(seq 0 "$count"); do
    pb_host_ip=$(jq -r ".mappings[$i] | .\"host-ip\"" "$PATH_PORTS_JSON")
    pb_host=$(jq -r ".mappings[$i] | .\"host-port\"" "$PATH_PORTS_JSON")
    pb_container=$(jq -r ".mappings[$i] | .\"container-port\"" "$PATH_PORTS_JSON")
    pb_protocol=$(jq -r ".mappings[$i] | .protocol" "$PATH_PORTS_JSON")
    publish_port "$pb_host_ip" "$pb_host" "$pb_container" "$pb_protocol"
    [ ! -z "$retval" ] && cat_data "$uuid" "$retval"
  done
fi

exit 0
